import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase"; // Removed .ts extension

export default class DeviceController {
  create = asyncHandler(async (req: Request, res: Response) => {
    console.log("hit devices");
    const {
      deviceName,
      macAddress, 
      operatingSystem,
      manufacturer,
      modelNumber,
      deviceType,
      notes,
      aircraftId,
      connectivity,
    } = req.body;

    if (!deviceName || !macAddress) {
      throw new BadRequestError('Device name and MAC address are required');
    }

    const docRef = await db.collection("devices").add({
      ...req.body,
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Device registered ✅',
      data: { 
        id: docRef.id,
        ...req.body 
      }
    });
  });

  getAll = asyncHandler(async (req: Request, res: Response) => {
    const snapshot = await db.collection("devices").orderBy("createdAt", "desc").get();
    const devices = snapshot.docs.map(doc => ({ id: doc.id, ...doc.data() }));
    console.log("devices hit:", devices)
    // Return direct array response
    return res.status(200).json(devices);
  });

  getOne = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const doc = await db.collection("devices").doc(id).get();
    if (!doc.exists) throw new BadRequestError('Device not found');
    
    // Return direct object response
    return res.status(200).json({ 
      id: doc.id, 
      ...doc.data() 
    });
  });

  update = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const deviceRef = db.collection("devices").doc(id);
    const doc = await deviceRef.get();
    
    if (!doc.exists) throw new BadRequestError('Device not found');

    await deviceRef.update({
      ...req.body,
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Device updated ✅'
    });
  });

  delete = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const deviceRef = db.collection("devices").doc(id);
    const doc = await deviceRef.get();
    
    if (!doc.exists) throw new BadRequestError('Device not found');
    
    await deviceRef.delete();
    
    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Device deleted ✅'
    });
  });

  // Optional: Add method to get devices by aircraft
  getByAircraft = asyncHandler(async (req: Request, res: Response) => {
    const { aircraftId } = req.params;
    
    const snapshot = await db.collection("devices")
      .where("aircraftId", "==", aircraftId)
      .orderBy("createdAt", "desc")
      .get();
    
    const devices = snapshot.docs.map(doc => ({ id: doc.id, ...doc.data() }));
    
    // Return direct array response
    return res.status(200).json(devices);
  });

  // Optional: Add method to get devices by type
  getByType = asyncHandler(async (req: Request, res: Response) => {
    const { deviceType } = req.params;
    
    const snapshot = await db.collection("devices")
      .where("deviceType", "==", deviceType)
      .orderBy("createdAt", "desc")
      .get();
    
    const devices = snapshot.docs.map(doc => ({ id: doc.id, ...doc.data() }));
    
    // Return direct array response
    return res.status(200).json(devices);
  });
}