import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase";

export default class ListController {
  // ====================== AIRPORTS ======================

  /** CREATE AIRPORT */
  createAirport = asyncHandler(async (req: Request, res: Response) => {
    const { code, name } = req.body;
    if (!code || !name) throw new BadRequestError("Code and name are required");

    const ref = db.collection("airports").doc();
    await ref.set({
      code,
      name,
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    return res.status(200).json({
      success: true,
      message: "Airport added",
      data: { id: ref.id },
    });
  });

  /** GET ALL AIRPORTS */
  getAirports = asyncHandler(async (_req: Request, res: Response) => {
    const snapshot = await db.collection("airports").get();
    const airports = snapshot.docs.map((doc) => ({ id: doc.id, ...doc.data() }));
    return res.status(200).json(airports);
  });

  /** DELETE AIRPORT */
  deleteAirport = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const docRef = db.collection("airports").doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError("Airport not found");
    }

    await docRef.delete();

    return res.status(200).json({
      success: true,
      message: "Airport deleted successfully",
    });
  });

  // ====================== REGNS (Derived from flights) ======================

  /** GET UNIQUE REGNs FROM FLIGHTS */
  getRegns = asyncHandler(async (_req: Request, res: Response) => {
    const snapshot = await db.collection("flights").orderBy("regn").get();

    const set = new Set<string>();
    snapshot.forEach((doc) => {
      const regn = doc.data().regn;
      if (regn) set.add(regn);
    });

    const regns = Array.from(set).map((regnNo) => ({ regnNo }));
    return res.status(200).json(regns);
  });

  // ====================== FLIGHT NUMBERS (Derived from flights) ======================

  /** GET UNIQUE FLIGHT NUMBERS FROM FLIGHTS */
  getFlightNos = asyncHandler(async (_req: Request, res: Response) => {
    const snapshot = await db.collection("flights").orderBy("fltNo").get();

    const set = new Set<string>();
    snapshot.forEach((doc) => {
      const fltNo = doc.data().fltNo;
      if (fltNo) set.add(fltNo);
    });

    const flightNos = Array.from(set).map((flightNo) => ({ flightNo }));
    return res.status(200).json(flightNos);
  });
}