import { Change } from '../../common/change';
import { DataSnapshot } from '../../common/providers/database';
import { CloudEvent, CloudFunction } from '../core';
import * as options from '../options';
import { Expression } from '../params';
export { DataSnapshot };
/** @hidden */
export interface RawRTDBCloudEventData {
    ['@type']: 'type.googleapis.com/google.events.firebase.database.v1.ReferenceEventData';
    data: any;
    delta: any;
}
/** @hidden */
export interface RawRTDBCloudEvent extends CloudEvent<RawRTDBCloudEventData> {
    firebasedatabasehost: string;
    instance: string;
    ref: string;
    location: string;
}
/** A CloudEvent that contains a DataSnapshot or a Change<DataSnapshot> */
export interface DatabaseEvent<T> extends CloudEvent<T> {
    /** The domain of the database instance */
    firebaseDatabaseHost: string;
    /** The instance ID portion of the fully qualified resource name */
    instance: string;
    /** The database reference path */
    ref: string;
    /** The location of the database */
    location: string;
    /**
     * An object containing the values of the path patterns.
     * Only named capture groups will be populated - {key}, {key=*}, {key=**}
     */
    params: Record<string, string>;
}
/** ReferenceOptions extend EventHandlerOptions with provided ref and optional instance  */
export interface ReferenceOptions extends options.EventHandlerOptions {
    /**
     * Specify the handler to trigger on a database reference(s).
     * This value can either be a single reference or a pattern.
     * Examples: '/foo/bar', '/foo/{bar}'
     */
    ref: string;
    /**
     * Specify the handler to trigger on a database instance(s).
     * If present, this value can either be a single instance or a pattern.
     * Examples: 'my-instance-1', 'my-instance-*'
     * Note: The capture syntax cannot be used for 'instance'.
     */
    instance?: string;
    /**
     * Region where functions should be deployed.
     */
    region?: options.SupportedRegion | string;
    /**
     * Amount of memory to allocate to a function.
     * A value of null restores the defaults of 256MB.
     */
    memory?: options.MemoryOption | Expression<number> | null;
    /**
     * Timeout for the function in sections, possible values are 0 to 540.
     * HTTPS functions can specify a higher timeout.
     * A value of null restores the default of 60s
     * The minimum timeout for a gen 2 function is 1s. The maximum timeout for a
     * function depends on the type of function: Event handling functions have a
     * maximum timeout of 540s (9 minutes). HTTPS and callable functions have a
     * maximum timeout of 36,00s (1 hour). Task queue functions have a maximum
     * timeout of 1,800s (30 minutes)
     */
    timeoutSeconds?: number | Expression<number> | null;
    /**
     * Min number of actual instances to be running at a given time.
     * Instances will be billed for memory allocation and 10% of CPU allocation
     * while idle.
     * A value of null restores the default min instances.
     */
    minInstances?: number | Expression<number> | null;
    /**
     * Max number of instances to be running in parallel.
     * A value of null restores the default max instances.
     */
    maxInstances?: number | Expression<number> | null;
    /**
     * Number of requests a function can serve at once.
     * Can only be applied to functions running on Cloud Functions v2.
     * A value of null restores the default concurrency (80 when CPU >= 1, 1 otherwise).
     * Concurrency cannot be set to any value other than 1 if `cpu` is less than 1.
     * The maximum value for concurrency is 1,000.
     */
    concurrency?: number | Expression<number> | null;
    /**
     * Fractional number of CPUs to allocate to a function.
     * Defaults to 1 for functions with <= 2GB RAM and increases for larger memory sizes.
     * This is different from the defaults when using the gcloud utility and is different from
     * the fixed amount assigned in Google Cloud Functions generation 1.
     * To revert to the CPU amounts used in gcloud or in Cloud Functions generation 1, set this
     * to the value "gcf_gen1"
     */
    cpu?: number | 'gcf_gen1';
    /**
     * Connect cloud function to specified VPC connector.
     * A value of null removes the VPC connector
     */
    vpcConnector?: string | null;
    /**
     * Egress settings for VPC connector.
     * A value of null turns off VPC connector egress settings
     */
    vpcConnectorEgressSettings?: options.VpcEgressSetting | null;
    /**
     * Specific service account for the function to run as.
     * A value of null restores the default service account.
     */
    serviceAccount?: string | null;
    /**
     * Ingress settings which control where this function can be called from.
     * A value of null turns off ingress settings.
     */
    ingressSettings?: options.IngressSetting | null;
    /**
     * User labels to set on the function.
     */
    labels?: Record<string, string>;
    secrets?: string[];
    /** Whether failed executions should be delivered again. */
    retry?: boolean | Expression<boolean> | null;
}
/**
 * Event handler which triggers when data is created, updated, or deleted in Realtime Database.
 *
 * @param reference - The database reference path to trigger on.
 * @param handler - Event handler which is run every time a Realtime Database create, update, or delete occurs.
 */
export declare function onValueWritten(reference: string, handler: (event: DatabaseEvent<Change<DataSnapshot>>) => any | Promise<any>): CloudFunction<DatabaseEvent<Change<DataSnapshot>>>;
/**
 * Event handler which triggers when data is created, updated, or deleted in Realtime Database.
 *
 * @param opts - Options that can be set on an individual event-handling function.
 * @param handler - Event handler which is run every time a Realtime Database create, update, or delete occurs.
 */
export declare function onValueWritten(opts: ReferenceOptions, handler: (event: DatabaseEvent<Change<DataSnapshot>>) => any | Promise<any>): CloudFunction<DatabaseEvent<Change<DataSnapshot>>>;
/**
 * Event handler which triggers when data is created in Realtime Database.
 *
 * @param reference - The database reference path to trigger on.
 * @param handler - Event handler which is run every time a Realtime Database create occurs.
 */
export declare function onValueCreated(reference: string, handler: (event: DatabaseEvent<DataSnapshot>) => any | Promise<any>): CloudFunction<DatabaseEvent<DataSnapshot>>;
/**
 * Event handler which triggers when data is created in Realtime Database.
 *
 * @param opts - Options that can be set on an individual event-handling function.
 * @param handler - Event handler which is run every time a Realtime Database create occurs.
 */
export declare function onValueCreated(opts: ReferenceOptions, handler: (event: DatabaseEvent<DataSnapshot>) => any | Promise<any>): CloudFunction<DatabaseEvent<DataSnapshot>>;
/**
 * Event handler which triggers when data is updated in Realtime Database.
 *
 * @param reference - The database reference path to trigger on.
 * @param handler - Event handler which is run every time a Realtime Database update occurs.
 */
export declare function onValueUpdated(reference: string, handler: (event: DatabaseEvent<Change<DataSnapshot>>) => any | Promise<any>): CloudFunction<DatabaseEvent<Change<DataSnapshot>>>;
/**
 * Event handler which triggers when data is updated in Realtime Database.
 *
 * @param opts - Options that can be set on an individual event-handling function.
 * @param handler - Event handler which is run every time a Realtime Database update occurs.
 */
export declare function onValueUpdated(opts: ReferenceOptions, handler: (event: DatabaseEvent<Change<DataSnapshot>>) => any | Promise<any>): CloudFunction<DatabaseEvent<Change<DataSnapshot>>>;
/**
 * Event handler which triggers when data is deleted in Realtime Database.
 *
 * @param reference - The database reference path to trigger on.
 * @param handler - Event handler which is run every time a Realtime Database deletion occurs.
 */
export declare function onValueDeleted(reference: string, handler: (event: DatabaseEvent<DataSnapshot>) => any | Promise<any>): CloudFunction<DatabaseEvent<DataSnapshot>>;
/**
 * Event handler which triggers when data is deleted in Realtime Database.
 *
 * @param opts - Options that can be set on an individual event-handling function.
 * @param handler - Event handler which is run every time a Realtime Database deletion occurs.
 */
export declare function onValueDeleted(opts: ReferenceOptions, handler: (event: DatabaseEvent<DataSnapshot>) => any | Promise<any>): CloudFunction<DatabaseEvent<DataSnapshot>>;
