import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase"; // Removed .ts extension

export default class AircraftController {
  create = asyncHandler(async (req: any, res: Response) => {
    const { aircraftName, registrationNumber, openingFlightHours } = req.body;
    const file = req.file;

    if (!aircraftName || !registrationNumber) {
      throw new BadRequestError('Aircraft name and registration number are required');
    }

    let imageUrl = "";
    if (file) imageUrl = `/uploads/${file.filename}`;

    const docRef = await db.collection("aircrafts").add({
      ...req.body,
      openingFlightHours: Number(openingFlightHours),
      imageUrl,
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Aircraft created successfully',
      data: {
        aircraftId: docRef.id,
        ...req.body,
        imageUrl
      }
    });
  });

  getAll = asyncHandler(async (req: Request, res: Response) => {
    const snapshot = await db.collection("aircrafts").orderBy("createdAt", "desc").get();
    const aircrafts = snapshot.docs.map(doc => ({
      id: doc.id,
      ...doc.data(),
    }));
    
    // Return direct array response
    return res.status(200).json(aircrafts);
  });

  getOne = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const doc = await db.collection("aircrafts").doc(id).get();
    if (!doc.exists) throw new BadRequestError('Aircraft not found');
    
    // Return direct object response
    return res.status(200).json({ 
      id: doc.id, 
      ...doc.data() 
    });
  });

  update = asyncHandler(async (req: any, res: Response) => {
    const { id } = req.params;
    const docRef = db.collection("aircrafts").doc(id);
    const doc = await docRef.get();
    if (!doc.exists) throw new BadRequestError('Aircraft not found');

    const updateData: any = {
      ...req.body,
      openingFlightHours: Number(req.body.openingFlightHours),
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    };

    if (req.file) updateData.imageUrl = `/uploads/${req.file.filename}`;

    await docRef.update(updateData);
    const updatedDoc = await docRef.get();
    
    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Aircraft updated successfully',
      data: { 
        id: updatedDoc.id, 
        ...updatedDoc.data() 
      }
    });
  });

  delete = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const docRef = db.collection("aircrafts").doc(id);
    const doc = await docRef.get();
    if (!doc.exists) throw new BadRequestError('Aircraft not found');
    await docRef.delete();
    
    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Aircraft deleted successfully'
    });
  });

  // Optional: Add method to get aircrafts by registration number
  getByRegistration = asyncHandler(async (req: Request, res: Response) => {
    const { registrationNumber } = req.params;
    
    const snapshot = await db.collection("aircrafts")
      .where("registrationNumber", "==", registrationNumber)
      .get();
    
    const aircrafts = snapshot.docs.map(doc => ({
      id: doc.id,
      ...doc.data(),
    }));
    
    // Return direct array response
    return res.status(200).json(aircrafts);
  });

  // Optional: Add method to update aircraft image only
  updateImage = asyncHandler(async (req: any, res: Response) => {
    const { id } = req.params;
    const file = req.file;

    if (!file) {
      throw new BadRequestError('No image file provided');
    }

    const docRef = db.collection("aircrafts").doc(id);
    const doc = await docRef.get();
    
    if (!doc.exists) {
      throw new BadRequestError('Aircraft not found');
    }

    const imageUrl = `/uploads/${file.filename}`;

    await docRef.update({
      imageUrl,
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Aircraft image updated successfully',
      data: { imageUrl }
    });
  });

  // Optional: Add method to get aircraft statistics
  getStats = asyncHandler(async (req: Request, res: Response) => {
    const snapshot = await db.collection("aircrafts").get();
    const aircrafts = snapshot.docs.map(doc => doc.data());
    
    const stats = {
      totalAircrafts: aircrafts.length,
      activeAircrafts: aircrafts.filter(a => a.status === 'active').length,
      maintenanceAircrafts: aircrafts.filter(a => a.status === 'maintenance').length,
      totalFlightHours: aircrafts.reduce((sum, aircraft) => sum + (aircraft.totalFlightHours || 0), 0),
    };

    // Return direct object response
    return res.status(200).json(stats);
  });
}