import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { SuccessResponse } from "../../utils/ApiResponse";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase";

export default class CheckController {
  
  saveChecks = asyncHandler(async (req: Request, res: Response) => {
    const { logId } = req.params;
    const { checks, createdBy } = req.body;

    if (!checks || !createdBy) {
      throw new BadRequestError("checks and createdBy are required");
    }

    const logRef = db.collection("logs").doc(logId);
    const logDoc = await logRef.get();
    if (!logDoc.exists) {
      throw new BadRequestError("Log not found");
    }

    const logData = logDoc.data()!;
    const logPageNo = logData.logPageNo;
    const flightLeg = logData.flightLeg || 0;

    const checkData = {
      logId,
      logPageNo,
      checks: Object.entries(checks).map(([type, details]: [string, any]) => ({
        type,
        authId: details.authId || "",
        authName: details.authName || "",
        authDate: details.authDate || "",
        svcOption: details.svcOption || "",
      })),
      createdBy,
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    };

    const existingCheckSnap = await db
      .collection("checks")
      .where("logId", "==", logId)
      .get();

    let checkId;
    if (existingCheckSnap.empty) {
      const checkRef = db.collection("checks").doc();
      checkId = checkRef.id;
      await checkRef.set(checkData);
    } else {
      checkId = existingCheckSnap.docs[0].id;
      await db.collection("checks").doc(checkId).update({
        checks: checkData.checks,
        updatedBy: createdBy,
        updatedAt: admin.firestore.FieldValue.serverTimestamp(),
      });
    }

    const hasAcceptance = checkData.checks.some((check: any) => check.type === "ACCEPTANCE");
    if (hasAcceptance && flightLeg === 0) {
      const logsSnap = await db.collection("logs").get();
      const batch = db.batch();

      for (const logDoc of logsSnap.docs) {
        const log = logDoc.data();
        const logRef = db.collection("logs").doc(logDoc.id);
        const flightRef = log.flightId ? db.collection("flights").doc(log.flightId) : null;

        let newFlightLeg = log.flightLeg;
        let newCurrentFlight = false;

        if (log.flightLeg === 0 && logDoc.id === logId) {
          newFlightLeg = -1;
          newCurrentFlight = false;
        } else if (log.flightLeg < 0) {
          newFlightLeg = log.flightLeg - 1;
        } else if (log.flightLeg > 0) {
          newFlightLeg = log.flightLeg - 1;
          if (newFlightLeg === 0) {
            newCurrentFlight = true;
          }
        }

        batch.update(logRef, {
          flightLeg: newFlightLeg,
          updatedAt: admin.firestore.FieldValue.serverTimestamp(),
        });

        if (flightRef) {
          batch.update(flightRef, {
            currentFlight: newCurrentFlight,
            updatedAt: admin.firestore.FieldValue.serverTimestamp(),
          });
        }
      }

      await batch.commit();
    }

    return new SuccessResponse("Check authorizations saved successfully ✅", {
      checkId,
      logId,
      logPageNo,
    }).send(res);
  });

  getChecks = asyncHandler(async (req: Request, res: Response) => {
    const { logId } = req.params;

    const checkSnap = await db
      .collection("checks")
      .where("logId", "==", logId)
      .get();

    if (checkSnap.empty) {
      return new SuccessResponse("Checks fetched", { checks: [] }).send(res);
    }

    const checkData = checkSnap.docs[0].data();
    return new SuccessResponse("Check authorizations fetched successfully", {
      checkId: checkSnap.docs[0].id,
      ...checkData,
    }).send(res);
  });
}