import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase"; // Removed .ts extension

export default class DeferralController {
  
  private validateDeferralEntry = (entry: any) => {
    if (!entry.description || !entry.description.trim()) {
      throw new BadRequestError('Description is required');
    }
    return true;
  };

  create = asyncHandler(async (req: Request, res: Response) => {
    const { entries } = req.body;
    
    if (!Array.isArray(entries)) {
      throw new BadRequestError('Entries must be an array');
    }

    const deferralsCollection = db.collection('deferrals');
    const results = [];

    for (const entry of entries) {
      this.validateDeferralEntry(entry);

      const typeNo = entry.defect_reference?.type_no;
      if (typeNo) {
        const existing = await deferralsCollection
          .where('entries.0.defect_reference.type_no', '==', typeNo)
          .get();
        if (!existing.empty) {
          throw new BadRequestError(`Type/No ${typeNo} already exists`);
        }
      }

      const docRef = deferralsCollection.doc();
      const deferralData = {
        entries: [{
          defect_reference: entry.defect_reference || {},
          description: entry.description,
          clear_reference: entry.clear_reference || {},
          enteredSign: entry.enteredSign || '',
          enteredAuth: entry.enteredAuth || '',
          enteredAuthName: entry.enteredAuthName || '',
          enteredDate: entry.enteredDate || '',
          expDate: entry.expDate || '',
          clearedSign: entry.clearedSign || '',
          clearedAuth: entry.clearedAuth || '',
          clearedAuthName: entry.clearedAuthName || '',
          clearedDate: entry.clearedDate || '',
          deferral: true,
          history: entry.history || [],
        }],
        createdAt: admin.firestore.FieldValue.serverTimestamp(),
        updatedAt: admin.firestore.FieldValue.serverTimestamp(),
      };

      await docRef.set(deferralData);

      await db.collection('deferralHistory').add({
        action: 'Created',
        createdAt: admin.firestore.FieldValue.serverTimestamp(),
        createdBy: entry.enteredAuth || 'Unknown',
        description: entry.description,
        ddNo: entry.defect_reference?.type_no || '',
        deferralId: docRef.id,
        logItemId: entry.defect_reference?.log_item_no || '',
        defect_reference: entry.defect_reference || {},
        log_page: entry.defect_reference?.log_page || '',
      });

      results.push({ id: docRef.id, message: 'Created ✅' });
    }

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Deferrals saved successfully ✅',
      data: { results }
    });
  });

  update = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const { entries } = req.body;
    
    if (!Array.isArray(entries) || entries.length !== 1) {
      throw new BadRequestError('Entries must be an array with exactly one entry');
    }

    const entry = entries[0];
    this.validateDeferralEntry(entry);

    const docRef = db.collection('deferrals').doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError('Deferral not found');
    }

    const deferralData = {
      entries: [{
        defect_reference: entry.defect_reference || {},
        description: entry.description,
        clear_reference: entry.clear_reference || {},
        enteredSign: entry.enteredSign || '',
        enteredAuth: entry.enteredAuth || '',
        enteredAuthName: entry.enteredAuthName || '',
        enteredDate: entry.enteredDate || '',
        expDate: entry.expDate || '',
        clearedSign: entry.clearedSign || '',
        clearedAuth: entry.clearedAuth || '',
        clearedAuthName: entry.clearedAuthName || '',
        clearedDate: entry.clearedDate || '',
        deferral: true,
        history: entry.history || [],
      }],
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    };

    await docRef.update(deferralData);

    await db.collection('deferralHistory').add({
      action: 'Updated',
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
      createdBy: entry.enteredAuth || 'Unknown',
      description: entry.description,
      ddNo: entry.defect_reference?.type_no || '',
      deferralId: id,
      logItemId: entry.defect_reference?.log_item_no || '',
      defect_reference: entry.defect_reference || {},
      log_page: entry.defect_reference?.log_page || '',
    });

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Updated ✅',
      data: { id }
    });
  });

  getAll = asyncHandler(async (req: Request, res: Response) => {
    const snapshot = await db.collection('deferrals').orderBy('ddNo', 'asc').get();
    const deferrals = snapshot.docs.map((doc) => ({ id: doc.id, ...doc.data() }));
    
    console.log("All Deferrals :", deferrals)
    // Return direct array response
    return res.status(200).json(deferrals);
  });

  getOne = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;

    const docRef = db.collection('deferrals').doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError('Deferral not found');
    }

    // Return direct object response
    return res.status(200).json({
      id: doc.id,
      ...doc.data()
    });
  });

  getHistoryByDeferralId = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;

    const snapshot = await db
      .collection('deferralHistory')
      .where('ddNo', '==', id)
      .get();

    if (snapshot.empty) {
      // Return direct array response
      return res.status(200).json([]);
    }

    const history = snapshot.docs.map((doc) => {
      const data = doc.data();
      return {
        id: doc.id,
        ...data,
        createdAt: data.createdAt && data.createdAt.toDate ? data.createdAt.toDate().toISOString() : null,
      };
    });

    // Return direct array response
    return res.status(200).json(history);
  });

  getHistoryByTypeNo = asyncHandler(async (req: Request, res: Response) => {
    const { typeNo } = req.params;

    const snapshot = await db
      .collection('deferralHistory')
      .where('ddNo', '==', typeNo)
      .orderBy('createdAt', 'desc')
      .get();

    if (snapshot.empty) {
      // Return direct array response
      return res.status(200).json([]);
    }

    const history = snapshot.docs.map((doc) => {
      const data = doc.data();
      return {
        id: doc.id,
        ...data,
        createdAt: data.createdAt && data.createdAt.toDate ? data.createdAt.toDate().toISOString() : null,
      };
    });

    // Return direct array response
    return res.status(200).json(history);
  });

  delete = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;

    const docRef = db.collection('deferrals').doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError('Deferral not found');
    }

    await docRef.delete();
    
    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Deferral deleted successfully ✅'
    });
  });

  // Optional: Add method to get deferrals by status
  getByStatus = asyncHandler(async (req: Request, res: Response) => {
    const { status } = req.params;

    const snapshot = await db.collection('deferrals')
      .where('entries.0.deferral', '==', status === 'active')
      .orderBy('updatedAt', 'asc')
      .get();

    const deferrals = snapshot.docs.map((doc) => ({ id: doc.id, ...doc.data() }));
    
    // Return direct array response
    return res.status(200).json(deferrals);
  });

  // Optional: Add method to clear a deferral
  clearDeferral = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const { clearedSign, clearedAuth, clearedAuthName, clearedDate } = req.body;

    const docRef = db.collection('deferrals').doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError('Deferral not found');
    }

    const data = doc.data();
    const entry = data?.entries?.[0];

    if (!entry) {
      throw new BadRequestError('Deferral entry not found');
    }

    await docRef.update({
      'entries.0.clearedSign': clearedSign || '',
      'entries.0.clearedAuth': clearedAuth || '',
      'entries.0.clearedAuthName': clearedAuthName || '',
      'entries.0.clearedDate': clearedDate || '',
      'entries.0.deferral': false,
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Deferral cleared successfully ✅'
    });
  });
}