import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase"; // Removed .ts extension

export default class NoticeController {
  
  private validateNoticeEntry = (entry: any) => {
    if (!entry.description || !entry.description.trim()) {
      throw new BadRequestError('Description is required');
    }
    return true;
  };

  replaceAll = asyncHandler(async (req: Request, res: Response) => {
    const { entries } = req.body;
    
    if (!Array.isArray(entries)) {
      throw new BadRequestError('Entries must be an array');
    }

    const noticesCollection = db.collection('notices');
    
    // Delete all existing notices
    const snapshot = await noticesCollection.get();
    const deletePromises: any[] = [];
    snapshot.forEach(doc => {
      deletePromises.push(noticesCollection.doc(doc.id).delete());
    });
    await Promise.all(deletePromises);

    // Insert new entries
    const results = [];
    for (const entry of entries) {
      this.validateNoticeEntry(entry);

      const docRef = noticesCollection.doc();
      const data = {
        entered_by_id: entry.entered_by_id || null,
        entered_by_name: entry.entered_by_name || null,
        cleared_by_id: entry.cleared_by_id || null,
        cleared_by_name: entry.cleared_by_name || null,
        description: entry.description,
        createdAt: admin.firestore.FieldValue.serverTimestamp(),
        updatedAt: admin.firestore.FieldValue.serverTimestamp(),
      };

      await docRef.set(data);
      results.push({ id: docRef.id, message: 'Created ✅' });
    }

    // Return direct object response
    return res.status(200).json({ 
      success: true,
      message: 'All notices replaced successfully ✅', 
      data: { results } 
    });
  });

  getAll = asyncHandler(async (req: Request, res: Response) => {
    const snapshot = await db
      .collection('notices')
      .orderBy('updatedAt', 'desc')
      .get();

    const notices = snapshot.docs.map((doc) => ({ id: doc.id, ...doc.data() }));
    console.log("fetched notices:", notices)
    // Return direct array response
    return res.status(200).json(notices);
  });

  deleteExpired = asyncHandler(async (req: Request, res: Response) => {
    const currentDate = new Date().toISOString().split('T')[0];
    const snapshot = await db
      .collection('notices')
      .where('expDate', '<=', currentDate)
      .get();

    const batch = db.batch();
    snapshot.docs.forEach((doc) => {
      batch.delete(doc.ref);
    });

    await batch.commit();
    
    // Return direct object response
    return res.status(200).json({ 
      success: true,
      message: `Deleted ${snapshot.docs.length} expired notices ✅`
    });
  });

  // Optional: Add create method for single notice
  create = asyncHandler(async (req: Request, res: Response) => {
    console.log("req.body:", req.body)
    const { entered_by_id, entered_by_name, cleared_by_id, cleared_by_name, description } = req.body;

    this.validateNoticeEntry({ description });

    const docRef = db.collection('notices').doc();
    const data = {
      entered_by_id: entered_by_id || null,
      entered_by_name: entered_by_name || null,
      cleared_by_id: cleared_by_id || null,
      cleared_by_name: cleared_by_name || null,
      description: description,
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    };

    await docRef.set(data);

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Notice created successfully ✅',
      data: { id: docRef.id }
    });
  });

  // Optional: Add delete method for single notice
  delete = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    
    const docRef = db.collection('notices').doc(id);
    const doc = await docRef.get();
    
    if (!doc.exists) {
      throw new BadRequestError('Notice not found');
    }

    await docRef.delete();

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Notice deleted successfully ✅'
    });
  });
}